<?php
/**
 * Elementor Properties Widget V13.
 *
 * Widget to display a carousel of properties, including an image, title,
 * excerpt, meta information (bed, bath, area), and price.
 *
 * @since 2.4.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Properties_Widget_V13 extends \Elementor\Widget_Base {

	use RHEA_Properties_Partials;
	use RHEA_Properties_Action_buttons;

	private $is_rvr_enabled;

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );

		$this->is_rvr_enabled = rhea_is_rvr_enabled();

		wp_register_script( 'rh-properties-widget-v13', RHEA_PLUGIN_URL . 'elementor/js/properties-widget-v13.js', [ 'elementor-frontend' ], RHEA_VERSION, true );
	}

	public function get_name() {
		return 'rh-properties-widget-v13';
	}

	public function get_title() {
		return esc_html__( 'RH: Properties V13 (Carousel)', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-post-slider rh-ultra-widget';
	}

	public function get_keywords() {
		return [ 'RealHomes', 'Ultra', 'Carousel', 'Properties Carousel' ];
	}

	public function get_categories() {
		return [ 'ultra-real-homes' ];
	}

	public function get_script_depends() {
		return [ 'rh-properties-widget-v13' ];
	}

	public function get_style_depends() {
		return [ 'swiper' ];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'properties_section',
			[
				'label' => esc_html__( 'Properties', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'widget_layout',
			[
				'label'   => esc_html__( 'Layout', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'simple'  => esc_html__( 'Simple', RHEA_TEXT_DOMAIN ),
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name'    => 'thumbnail',
				'default' => 'post-featured-image',
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'   => esc_html__( 'Number of Properties', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 50,
				'step'    => 1,
				'default' => 5,
			]
		);

		$this->add_control(
			'featured_properties_condition',
			[
				'label'   => esc_html__( 'Featured Properties', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'normal'             => esc_html__( 'Normal', RHEA_TEXT_DOMAIN ),
					'show_featured_only' => esc_html__( 'Show Featured Only', RHEA_TEXT_DOMAIN ),
					'hide_featured'      => esc_html__( 'Hide Featured', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'normal',
			]
		);

		// Property taxonomies controls
		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$options = [];
				$terms   = get_terms( $single_tax->name );

				if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
					foreach ( $terms as $term ) {
						$options[ $term->slug ] = $term->name;
					}
				}

				$this->add_control(
					$single_tax->name,
					[
						'label'       => $single_tax->label,
						'type'        => \Elementor\Controls_Manager::SELECT2,
						'multiple'    => true,
						'label_block' => true,
						'options'     => $options,
					]
				);
			}
		}

		$this->add_control(
			'order',
			[
				'label'   => esc_html__( 'Order', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'asc'  => esc_html__( 'Ascending', RHEA_TEXT_DOMAIN ),
					'desc' => esc_html__( 'Descending', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'desc',
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'   => esc_html__( 'Order By', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'date'       => esc_html__( 'Date', RHEA_TEXT_DOMAIN ),
					'price'      => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
					'title'      => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
					'menu_order' => esc_html__( 'Menu Order', RHEA_TEXT_DOMAIN ),
					'rand'       => esc_html__( 'Random', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'date',
			]
		);

		$this->add_control(
			'show_old_price',
			[
				'label'        => esc_html__( 'Show Old Price', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'        => esc_html__( 'Show Excerpt', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',

			]
		);

		$this->add_control(
			'excerpt_length',
			[
				'label'     => esc_html__( 'Excerpt Length (Words)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'min'       => 5,
				'max'       => 100,
				'default'   => 15,
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$allowed_html = array(
			'a'      => array(
				'href'  => array(),
				'title' => array()
			),
			'br'     => array(),
			'em'     => array(),
			'strong' => array(),
		);

		$this->add_control(
			'ere_enable_fav_properties',
			[
				'label'        => esc_html__( 'Add To Favourite', RHEA_TEXT_DOMAIN ),
				'description'  => wp_kses( __( '<strong>Important:</strong> Make sure to select <strong>Show</strong> in Customizer <strong>Dashboard > My Favorites (Add to Favorites Button on Property Detail Page)</strong> settings. ', RHEA_TEXT_DOMAIN ), $allowed_html ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'widget_layout' => 'simple',
				],
			]
		);

		$this->add_control(
			'ere_enable_compare_properties',
			[
				'label'        => esc_html__( 'Add To Compare', RHEA_TEXT_DOMAIN ),
				'description'  => wp_kses( __( '<strong>Important:</strong> Make sure to select <strong>Enable</strong> in Customizer <strong>Floating Features (Compare Properties)</strong> settings. ', RHEA_TEXT_DOMAIN ), $allowed_html ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'widget_layout' => 'simple',
				],
			]
		);

        $this->add_control(
            'show_animate_contents',
            [
                'label'        => esc_html__( 'Show Animate Contents (Editor Mode Only)', RHEA_TEXT_DOMAIN ),
                'description'  => esc_html__( 'Only for Elementor Editor to stylize the contents', RHEA_TEXT_DOMAIN ),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
                'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
                'return_value' => 'yes',
                'default'      => 'no',
            ]
        );

		$this->end_controls_section();

		$this->start_controls_section(
			'ere_properties_labels',
			[
				'label'     => esc_html__( 'Labels', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'widget_layout' => 'simple',
				],
			]
		);

		$this->action_buttons_labels();

		$this->end_controls_section();

		$this->start_controls_section(
			'meta_settings',
			[
				'label'     => esc_html__( 'Meta Settings', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->add_control(
			'meta_layout',
			[
				'label'   => esc_html__( 'Layout', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => '2',
				'options' => [
					'1' => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'2' => esc_html__( 'Simple', RHEA_TEXT_DOMAIN ),
				],
			]
		);

		$default_meta_fields = array(
			'bedrooms'   => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			'bathrooms'  => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			'area'       => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			'garage'     => esc_html__( 'Garages/Parking', RHEA_TEXT_DOMAIN ),
			'year-built' => esc_html__( 'Year Built', RHEA_TEXT_DOMAIN ),
			'lot-size'   => esc_html__( 'Lot Size', RHEA_TEXT_DOMAIN ),
		);

		$repeater_meta_fields = array(
			array(
				'rhea_property_meta_display' => 'bedrooms',
				'rhea_meta_repeater_label'   => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			),
			array(
				'rhea_property_meta_display' => 'bathrooms',
				'rhea_meta_repeater_label'   => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			),
			array(
				'rhea_property_meta_display' => 'area',
				'rhea_meta_repeater_label'   => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			),
		);

		if ( $this->is_rvr_enabled ) {
			$default_meta_fields['guests']   = esc_html__( 'Guests Capacity', RHEA_TEXT_DOMAIN );
			$default_meta_fields['min-stay'] = esc_html__( 'Min Stay', RHEA_TEXT_DOMAIN );

			$repeater_meta_fields[] = array(
				'rhea_property_meta_display' => 'guests',
				'rhea_meta_repeater_label'   => esc_html__( 'Guests', RHEA_TEXT_DOMAIN ),
			);
		}

		$default_meta_fields = apply_filters( 'rhea_custom_fields_meta_icons', $default_meta_fields );

		$meta_repeater = new \Elementor\Repeater();

		$meta_repeater->add_control(
			'rhea_property_meta_display',
			[
				'label'   => esc_html__( 'Select Meta', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => '',
				'options' => $default_meta_fields,
			]
		);

		$meta_repeater->add_control(
			'rhea_meta_repeater_label',
			[
				'label'   => esc_html__( 'Meta Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Add Label', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'rhea_add_meta_select',
			[
				'label'       => esc_html__( 'Add Meta', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::REPEATER,
				'fields'      => $meta_repeater->get_controls(),
				'default'     => $repeater_meta_fields,
				'title_field' => ' {{{ rhea_meta_repeater_label }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'slider_settings',
			[
				'label' => esc_html__( 'Slider Settings', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'thumbnails_slider_effect',
			[
				'label'              => esc_html__( 'Thumbnails Slider Effect', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'default'            => 'slide',
				'options'            => [
					'slide' => esc_html__( 'Slide', RHEA_TEXT_DOMAIN ),
					'fade'  => esc_html__( 'Fade', RHEA_TEXT_DOMAIN ),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'contents_slider_effect',
			[
				'label'              => esc_html__( 'Contents Slider Effect', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SELECT,
				'default'            => 'fade',
				'options'            => [
					'slide' => esc_html__( 'Slide', RHEA_TEXT_DOMAIN ),
					'fade'  => esc_html__( 'Fade', RHEA_TEXT_DOMAIN ),
				],
				'frontend_available' => true,
				'condition'          => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label'              => esc_html__( 'Autoplay', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'pause_on_hover',
			[
				'label'              => esc_html__( 'Pause on Hover', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'condition'          => [
					'autoplay' => 'yes',
				],
				'render_type'        => 'none',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'pause_on_interaction',
			[
				'label'              => esc_html__( 'Pause on Interaction', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'condition'          => [
					'autoplay' => 'yes',
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'infinite',
			[
				'label'              => esc_html__( 'Infinite Loop', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label'              => esc_html__( 'Autoplay Speed', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'default'            => 4000,
				'condition'          => [
					'autoplay' => 'yes',
				],
				'render_type'        => 'none',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'speed',
			[
				'label'              => esc_html__( 'Animation Speed', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::NUMBER,
				'default'            => 500,
				'render_type'        => 'none',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'thumbnails_section',
			[
				'label' => esc_html__( 'Thumbnails Slider', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'thumbnails_slider_image_position',
			[
				'label'     => esc_html__( 'Image Position', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					''       => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'top'    => esc_html__( 'Top', RHEA_TEXT_DOMAIN ),
					'bottom' => esc_html__( 'Bottom', RHEA_TEXT_DOMAIN ),
					'center' => esc_html__( 'Center', RHEA_TEXT_DOMAIN ),
					'left'   => esc_html__( 'Left', RHEA_TEXT_DOMAIN ),
					'right'  => esc_html__( 'Right', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .property-image img' => 'object-position: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'thumbnails_slider_height',
			[
				'label'     => esc_html__( 'Height', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .property-image,
					 {{WRAPPER}} .rh-properties-widget-v13 .property-image img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'thumbnails_slider_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .thumbnails-slider,
					 {{WRAPPER}} .rh-properties-widget-v13 .thumbnails-slider .property-image,
					 {{WRAPPER}} .rh-properties-widget-v13 .thumbnails-slider .property-image img' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'thumbnails_slider_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .thumbnails-slider' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'content_container_section',
			[
				'label'     => esc_html__( 'Content Wrapper', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'condition' => [
					'widget_layout' => 'simple',
				],
			]
		);

		$this->add_responsive_control(
			'content_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13-simple-layout .property-item-content-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'widget_layout' => 'simple',
				],
			]
		);

		$this->add_responsive_control(
			'content_container_background',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13-simple-layout .property-item-content-wrapper' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'widget_layout' => 'simple',
				],
			]
		);

		$this->end_controls_section();

		$this->properties_tags_controls();

		$this->start_controls_section(
			'property_title_section',
			[
				'label' => esc_html__( 'Property Title', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .property-title a',
			]
		);

		$this->add_control(
			'property_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .property-title a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_title_hover_color',
			[
				'label'     => esc_html__( 'Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .property-title a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_title_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .property-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'text_overflow',
			[
				'label'   => esc_html__( 'Text Over Flow', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 'eclipse',
				'options' => [
					'default' => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'eclipse' => esc_html__( 'Eclipse', RHEA_TEXT_DOMAIN ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_excerpt_section',
			[
				'label' => esc_html__( 'Property Excerpt', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_excerpt_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .property-excerpt',
			]
		);

		$this->add_control(
			'property_excerpt_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .property-excerpt' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_excerpt_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .property-excerpt' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_meta_section',
			[
				'label'     => esc_html__( 'Property Meta', RHEA_TEXT_DOMAIN ),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->add_responsive_control(
			'property_meta_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card_meta_wrap_stylish' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '
				{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rh_prop_card__meta .rhea_meta_titles, 
				{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rh_prop_card__meta .figure,
				{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rhea_ultra_prop_card__meta .rhea-meta-icons-labels, 
				{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rhea_ultra_prop_card__meta .figure
				',
			]
		);

		$this->add_control(
			'property_meta_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'
					{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rh_prop_card__meta .rhea_meta_titles,
					{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rh_prop_card__meta .figure,
					{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rhea_ultra_prop_card__meta .rhea-meta-icons-labels, 
				    {{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rhea_ultra_prop_card__meta .figure					
					' => 'color: {{VALUE}}
				',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_postfix_typography',
				'label'    => esc_html__( 'Postfix Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rh_prop_card__meta .label',
			]
		);

		$this->add_control(
			'property_meta_postfix_color',
			[
				'label'     => esc_html__( 'Postfix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh_prop_card_meta_wrap_stylish .rh_prop_card__meta .label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_color',
			[
				'label'     => esc_html__( 'SVG Icon Fill', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta svg,
					{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta .rhea_guests,
					{{WRAPPER}} .rh-properties-widget-v13 .rhea_ultra_prop_card__meta svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_stroke',
			[
				'label'     => esc_html__( 'SVG Icon Stroke', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta svg path,
					{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta svg rect,
					{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta .rhea_guests path,
					{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta .rhea_guests rect,
					{{WRAPPER}} .rh-properties-widget-v13 .rhea_ultra_prop_card__meta svg path,
					{{WRAPPER}} .rh-properties-widget-v13 .rhea_ultra_prop_card__meta svg rect' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'property_meta_icon_size',
			[
				'label'     => esc_html__( 'Meta Icon Size (px)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .rh_prop_card__meta svg'         => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-properties-widget-v13 .rhea_ultra_prop_card__meta svg' => 'width: {{SIZE}}{{UNIT}}; height:auto;',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'property_price_section',
			[
				'label' => esc_html__( 'Property Price', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-properties-widget-v13 .property-current-price, {{WRAPPER}} .rh-properties-widget-v13 .ere-price-display',
			]
		);

		$this->add_control(
			'property_price_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .property-current-price, {{WRAPPER}} .rh-properties-widget-v13 .ere-price-display' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'      => 'property_old_price_typography',
				'label'     => esc_html__( 'Old Price, Pre, and Postfix', RHEA_TEXT_DOMAIN ),
				'global'    => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'  => '{{WRAPPER}} .rh-properties-widget-v13 .property-old-price,
				 {{WRAPPER}} .rh-properties-widget-v13 .ere-price-prefix, 
				 {{WRAPPER}} .rh-properties-widget-v13 .ere-price-slash, 
				 {{WRAPPER}} .rh-properties-widget-v13 .ere-price-postfix',
				'condition' => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->add_control(
			'property_old_price_color',
			[
				'label'     => esc_html__( 'Old Price, Pre, and Postfix Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-properties-widget-v13 .property-old-price,
				 {{WRAPPER}} .rh-properties-widget-v13 .ere-price-prefix, 
				 {{WRAPPER}} .rh-properties-widget-v13 .ere-price-slash, 
				 {{WRAPPER}} .rh-properties-widget-v13 .ere-price-postfix' => 'color: {{VALUE}}',
				],
				'condition' => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pagination_section',
			[
				'label' => esc_html__( 'Pagination', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'pagination_bg_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled span' => '--swiper-pagination-bullet-inactive-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'pagination_active_bullet_bg_color',
			[
				'label'     => esc_html__( 'Active Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled span.swiper-pagination-bullet-active,
					 {{WRAPPER}} .swiper-pagination-styled span:hover'            => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .swiper-pagination-styled .select-active-bullet' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'pagination_container_section',
			[
				'label' => esc_html__( 'Container Options', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'pagination_container_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_container_position',
			[
				'label'     => esc_html__( 'Position', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'static',
				'options'   => [
					''         => esc_html__( 'Default', RHEA_TEXT_DOMAIN ),
					'static'   => esc_html__( 'Static', RHEA_TEXT_DOMAIN ),
					'absolute' => esc_html__( 'Absolute', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled' => 'position: {{VALUE}};',
				],
				'condition' => [
					'widget_layout' => 'default',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_container_position_left',
			[
				'label'     => esc_html__( 'Left Position', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => -1000,
						'max' => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled' => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'widget_layout'                 => 'default',
					'pagination_container_position' => 'absolute',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_container_position_bottom',
			[
				'label'     => esc_html__( 'Bottom Position', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => -1000,
						'max' => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled' => 'bottom: {{SIZE}}{{UNIT}} !important;',
				],
				'condition' => [
					'widget_layout'                 => 'default',
					'pagination_container_position' => 'absolute',
				],
			]
		);

		$this->add_control(
			'pagination_items_options_section',
			[
				'label' => esc_html__( 'Items Options', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'pagination_items_gap',
			[
				'label'     => esc_html__( 'Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-inner' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_items_width',
			[
				'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .swiper-pagination-styled .select-active-bullet'     => 'width: calc({{SIZE}}{{UNIT}} + 6px);',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_items_height',
			[
				'label'     => esc_html__( 'Height', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-bullet' => 'height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .swiper-pagination-styled .select-active-bullet'     => 'height: calc({{SIZE}}{{UNIT}} + 6px);',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_items_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-styled .swiper-pagination-bullet, {{WRAPPER}} .swiper-pagination-styled .select-active-bullet' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->action_button_controls_v2();
	}

	protected function render() {
		global $widget_id, $settings;

		$settings      = $this->get_settings_for_display();
		$widget_id     = $this->get_id();
		$widget_layout = $settings['widget_layout'];
		$meta_query = array();

		if ( ! $settings['posts_per_page'] ) {
			$settings['posts_per_page'] = 5;
		}

		$properties_args = array(
			'post_type'      => 'property',
			'posts_per_page' => $settings['posts_per_page'],
			'order'          => $settings['order'],
			'post_status'    => 'publish',
		);

		// Featured properties condition
		if ( ! empty( $settings['featured_properties_condition'] ) ) {
			$condition = $settings['featured_properties_condition'];

			if ( 'show_featured_only' === $condition ) {
				$meta_query[] = [
					'key'     => 'REAL_HOMES_featured',
					'value'   => 1,
					'compare' => '=',
					'type'    => 'NUMERIC',
				];
			} else if ( 'hide_featured' === $condition ) {
				$meta_query[] = [
					'key'     => 'REAL_HOMES_featured',
					'value'   => 1,
					'compare' => '!=',
					'type'    => 'NUMERIC',
				];
			}
		}

		// Add meta_query if not empty
		if ( ! empty( $meta_query ) ) {
			$properties_args['meta_query'] = $meta_query;
		}

		if ( 'price' === $settings['orderby'] ) {
			$properties_args['orderby']  = 'meta_value_num';
			$properties_args['meta_key'] = 'REAL_HOMES_property_price';
		} else {
			// For date, title, menu_order and rand
			$properties_args['orderby'] = $settings['orderby'];
		}

		// Filter based on custom taxonomies
		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$setting_key = $single_tax->name;
				if ( ! empty( $settings[ $setting_key ] ) ) {
					$properties_args['tax_query'][] = [
						'taxonomy' => $setting_key,
						'field'    => 'slug',
						'terms'    => $settings[ $setting_key ],
					];
				}
			}

			if ( isset( $properties_args['tax_query'] ) && count( $properties_args['tax_query'] ) > 1 ) {
				$properties_args['tax_query']['relation'] = 'AND';
			}
		}

		$properties_args  = apply_filters( 'rh_properties_widget_args', $properties_args );
		$properties_query = new WP_Query( $properties_args );

		$show_contents = '';
		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() && 'yes' === $settings['show_animate_contents'] ) {
			$show_contents = ' animate-in ';
		}

		if ( $properties_query->have_posts() ) {

			$thumbnails_slider = '';
			$contents_slider   = '';

			if ( $properties_query->have_posts() ) {
				while ( $properties_query->have_posts() ) {
					$properties_query->the_post();

					$property_id    = get_the_ID();
					$property_title = get_the_title( $property_id );
					$property_link  = get_the_permalink( $property_id );

					if ( has_post_thumbnail( $property_id ) ) {
						$property_thumbnail = get_the_post_thumbnail( $property_id, $settings['thumbnail_size'] );
					} else {
						$property_thumbnail = get_inspiry_image_placeholder( $settings['thumbnail_size'] );
					}

					if ( 'simple' === $widget_layout ) {

						$thumbnails_slider_content = '<article class="property-item rhea-animate-item position-bottom deanimate-sm-devices'. esc_attr( $show_contents ) .'">';
						$thumbnails_slider_content .= '<div class="property-item-content-wrapper">';
						$thumbnails_slider_content .= '<div class="property-item-inner">';

						if ( $property_title ) {
							$thumbnails_slider_content .= sprintf( '<header class="property-header"><h3 class="property-title"><a href="%s">%s</a></h3></header>', esc_url( $property_link ), esc_html( $property_title ) );
						}

						if ( 'yes' == $settings['show_excerpt'] ) {
							$thumbnails_slider_content .= sprintf( '<div class="property-excerpt"><p>%s</p></div>', esc_html( rhea_get_framework_excerpt( intval( $settings['excerpt_length'] ) ) ) );
						}

						$thumbnails_slider_content .= '<div class="rh-property-actions">';
						ob_start();
						rhea_render_favourite_button( $property_id, $settings, '/common/images/icons/ultra-favourite.svg', 'ultra' );
						rhea_render_compare_button( $property_id, $settings );
						$thumbnails_slider_content .= ob_get_clean();
						$thumbnails_slider_content .= '</div>';

						$thumbnails_slider_content .= '</div>';

						if ( function_exists( 'ere_property_price' ) ) {
							$thumbnails_slider_content .= sprintf( '<footer class="property-footer">%s</footer>', ere_get_property_price( $property_id, false, true ) );
						}

						$thumbnails_slider_content .= '</div>';
						$thumbnails_slider_content .= '</article>';

						$thumbnails_slider .= '<div class="swiper-slide"><figure class="property-image rhea-trigger-animate">';

						$thumbnails_slider .= sprintf( '<a href="%s">%s</a>%s', esc_url( $property_link ), $property_thumbnail, $thumbnails_slider_content );

						ob_start();
						rhea_render_properties_tags( $property_id, $settings, 'rhea-animate-item top deanimate-top-sm-devices' . $show_contents );
						$thumbnails_slider .= ob_get_clean();

						$thumbnails_slider .= '</figure></div>';

					} else {
						$thumbnails_slider .= '<div class="swiper-slide"><figure class="property-image rhea-trigger-animate">';
						$thumbnails_slider .= sprintf( '<a href="%s">%s</a>', esc_url( $property_link ), $property_thumbnail );

						ob_start();
						rhea_render_properties_tags( $property_id, $settings, 'rhea-animate-item top deanimate-top-sm-devices' . $show_contents );
						$thumbnails_slider .= ob_get_clean();

						$thumbnails_slider .= '</figure></div>';

						$contents_slider .= '<div class="swiper-slide"><article class="property-item">';

						if ( $property_title ) {
							$text_overflow   = 'rhea-text-eclipse-' . $settings['text_overflow'];
							$contents_slider .= sprintf( '<header class="property-header"><h3 class="property-title %s"><a href="%s">%s</a></h3></header>', esc_attr( $text_overflow ), esc_url( $property_link ), esc_html( $property_title ) );
						}

						if ( 'yes' == $settings['show_excerpt'] ) {
							$contents_slider .= sprintf( '<div class="property-excerpt"><p>%s</p></div>', esc_html( rhea_get_framework_excerpt( intval( $settings['excerpt_length'] ) ) ) );
						}

						ob_start();
						rhea_get_template_part( 'assets/partials/stylish/grid-card-meta' );
						$contents_slider .= ob_get_clean();

						if ( function_exists( 'ere_property_price' ) ) {
							$show_old_price  = ( 'yes' === $settings['show_old_price'] );
							$contents_slider .= sprintf( '<footer class="property-footer">%s</footer>', ere_get_property_price( $property_id, $show_old_price, true ) );
						}

						$contents_slider .= '</article></div>';
					}
				}

				wp_reset_postdata();
			}

			$container_classes[] = 'rh-properties-widget-v13-' . $widget_layout . '-layout';
			$container_classes[] = 'meta-layout-' . $settings['meta_layout'];
			$is_rtl              = is_rtl() ? 'dir="rtl"' : '';
			?>
            <section id="rh-properties-widget-v13-<?php echo esc_attr( $widget_id ); ?>" class="rh-properties-widget-v13 <?php echo join( " ", $container_classes ); ?>">
				<?php
				if ( ! empty( $thumbnails_slider ) ) {
					?>
                    <div id="thumbnails-slider-<?php echo esc_attr( $widget_id ); ?>" class="thumbnails-slider swiper swiper-container" <?php echo esc_attr( $is_rtl ); ?>>
                        <div class="swiper-wrapper">
							<?php echo $thumbnails_slider; ?>
                        </div>
                    </div>

                    <div id="swiper-pagination-<?php echo esc_attr( $widget_id ); ?>" class="swiper-pagination swiper-pagination-styled">
                        <div class="swiper-pagination-inner"></div>
                    </div>
					<?php
				}

				if ( ! empty( $contents_slider ) && 'default' === $widget_layout ) {
					?>
                    <div id="contents-slider-<?php echo esc_attr( $widget_id ); ?>" class="contents-slider swiper swiper-container" <?php echo esc_attr( $is_rtl ); ?>>
                        <div class="swiper-wrapper">
							<?php echo $contents_slider; ?>
                        </div>
                    </div>
					<?php
				}
				?>
            </section>
			<?php
		}
	}
}
