<?php
/**
 * Single Property Meta Widget V2.
 *
 * @since 2.4.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Meta_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	private $is_rvr_enabled;

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );

		$this->is_rvr_enabled = rhea_is_rvr_enabled();
	}

	public function get_name() {
		return 'rh-single-property-meta-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Meta V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-product-meta rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	public function get_keywords() {
		return [ 'RealHomes', 'Beds', 'Bath', 'Parking', 'Bedrooms', 'Bathroom', 'Area' ];
	}

	protected function register_controls() {

		$this->title_section_content_tab_controls();

		$this->start_controls_section(
			'property_meta_section',
			[
				'label' => esc_html__( 'Property Meta', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$meta_keys_list = array(
			'bedrooms'    => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
			'bathrooms'   => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
			'area'        => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
			'garage'      => esc_html__( 'Garages', RHEA_TEXT_DOMAIN ),
			'year-built'  => esc_html__( 'Year Built', RHEA_TEXT_DOMAIN ),
			'lot-size'    => esc_html__( 'Lot Size', RHEA_TEXT_DOMAIN ),
			'property-id' => esc_html__( 'Property ID', RHEA_TEXT_DOMAIN ),
		);

		$default_meta = array(
			array(
				'meta'       => 'bedrooms',
				'meta_label' => $meta_keys_list['bedrooms'],
			),
			array(
				'meta'       => 'bathrooms',
				'meta_label' => $meta_keys_list['bathrooms'],
			),
			array(
				'meta'       => 'area',
				'meta_label' => $meta_keys_list['area'],
			),
			array(
				'meta'       => 'garage',
				'meta_label' => $meta_keys_list['garage'],
			),
			array(
				'meta'       => 'year-built',
				'meta_label' => $meta_keys_list['year-built'],
			),
			array(
				'meta'       => 'lot-size',
				'meta_label' => $meta_keys_list['lot-size'],
			),
		);

		if ( $this->is_rvr_enabled ) {
			$meta_keys_list['guests']   = esc_html__( 'Guests Capacity', RHEA_TEXT_DOMAIN );
			$meta_keys_list['min-stay'] = esc_html__( 'Min Stay', RHEA_TEXT_DOMAIN );

			$default_meta[] = array(
				'meta'       => 'guests',
				'meta_label' => $meta_keys_list['guests'],
			);

			$default_meta[] = array(
				'meta'       => 'min-stay',
				'meta_label' => $meta_keys_list['min-stay'],
			);
		}

		$meta_repeater = new \Elementor\Repeater();
		$meta_repeater->add_control(
			'meta',
			[
				'label'   => esc_html__( 'Select Meta', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => $meta_keys_list,
				'default' => '',
			]
		);
		$meta_repeater->add_control(
			'meta_label',
			[
				'label'   => esc_html__( 'Meta Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Add Label', RHEA_TEXT_DOMAIN ),
			]
		);
		$meta_repeater->add_control(
			'meta_icon',
			[
				'label'       => esc_html__( 'Custom Icon', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
			]
		);

		$this->add_control(
			'selected_meta_list',
			[
				'label'       => esc_html__( 'Add Meta', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::REPEATER,
				'fields'      => $meta_repeater->get_controls(),
				'default'     => $default_meta,
				'title_field' => ' {{{ meta_label }}}',
			]
		);

		$this->end_controls_section();

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'property_meta_style_section',
			[
				'label' => esc_html__( 'Property Meta', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'meta_items_heading',
			[
				'label' => esc_html__( 'Meta Items', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_responsive_control(
			'meta_items_gap',
			[
				'label'     => esc_html__( 'Column Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2-inner' => 'column-gap: {{SIZE}}{{UNIT}};  --meta-v2-item-column-gap: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'meta_items_row_gap',
			[
				'label'     => esc_html__( 'Row Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2-inner' => 'row-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_items_width',
			[
				'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item' => 'flex: 0 0 calc({{SIZE}}{{UNIT}} - var(--meta-v2-item-column-gap, 44px));  max-width: calc({{SIZE}}{{UNIT}} - var(--meta-v2-item-column-gap, 44px));',
				],
			]
		);

		$this->add_control(
			'meta_label_heading',
			[
				'label'     => esc_html__( 'Meta Label', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'meta_label_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_label_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-label',
			]
		);

		$this->add_control(
			'meta_value_heading',
			[
				'label'     => esc_html__( 'Meta Value', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'meta_value_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-value' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_value_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-value',
			]
		);

		$this->add_control(
			'meta_value_postfix_heading',
			[
				'label'     => esc_html__( 'Meta Value Postfix', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'meta_value_postfix',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-value-postfix' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_value_postfix_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-value-postfix',
			]
		);

		$this->add_control(
			'meta_icon_heading',
			[
				'label'     => esc_html__( 'Meta Icon', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'property_meta_svg_color',
			[
				'label'     => esc_html__( 'SVG Icon Fill', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-icon-wrapper svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'property_meta_svg_stroke',
			[
				'label'     => esc_html__( 'SVG Icon Stroke', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-icon-wrapper svg path,
					{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-icon-wrapper svg rect' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'meta_icon_size',
			[
				'label'     => esc_html__( 'Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-icon-wrapper svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-meta-v2 .meta-item-icon-wrapper img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();

		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() || empty( $settings['selected_meta_list'] ) ) {
			return;
		}

		$property_id = get_the_ID();
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			// Sample property id for editor side,
			$property_id = rhea_get_sample_property_id();
		}
		?>
        <div id="rh-single-property-meta-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-meta-v2">
			<?php
			if ( ! empty( $settings['section_title'] ) ) {
				?>
                <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
				<?php
			}
			?>
            <div class="rh-single-property-meta-v2-inner">
				<?php
				$meta_keys_list     = rhea_get_property_meta_keys_list();
				$selected_meta_list = $settings['selected_meta_list'];

				foreach ( $selected_meta_list as $property_meta ) {

					// Skip this iteration if the selected meta is either empty or not defined in our meta keys list.
					if ( empty( $property_meta['meta'] ) || ! isset( $meta_keys_list[ $property_meta['meta'] ] ) ) {
						continue;
					}

					// Extract meta settings defined by user/admin and from global meta_keys_list mapping.
					$selected_meta_name  = $property_meta['meta'];
					$selected_meta_label = $property_meta['meta_label'] ?? '';
					$selected_meta_icon  = $property_meta['meta_icon'] ?? [];
					$meta_data           = $meta_keys_list[ $selected_meta_name ] ?? [];

					// Each entry in meta_keys_list typically contains:
					//   - 'key'     => actual property meta key to fetch
					//   - 'postfix' => postfix meta key (optional, e.g., size_unit)
					//   - 'icon'    => default icon
					$meta_key         = $meta_data['key'] ?? '';
					$meta_postfix_key = $meta_data['postfix'] ?? '';

					/**
					 * Determine postfix (e.g., units like "sq ft", "acres").
					 *
					 * For property size/lot size, helper functions return the *unit string* directly.
					 * We cannot treat these as meta keys, so we assign the returned string to $postfix.
					 */
					$postfix = '';
					if ( 'REAL_HOMES_property_size' === $meta_key ) {
						$postfix = realhomes_get_area_unit( $property_id );
					} else if ( 'REAL_HOMES_property_lot_size' === $meta_key ) {
						$postfix = realhomes_get_lot_unit( $property_id );
					}

					// If helper didn’t provide a unit, fall back to the postfix meta key (if defined).
					if ( empty( $postfix ) && ! empty( $meta_postfix_key ) ) {
						$postfix = get_post_meta( $property_id, $meta_postfix_key, true );
					}

					// Resolve the icon: custom icon (from user/admin) takes precedence over default.
					$default_icon   = $meta_data['icon'] ?? '';
					$meta_value     = $meta_key ? get_post_meta( $property_id, $meta_key, true ) : '';
					$is_custom_icon = ! empty( $selected_meta_icon['value'] );
					$meta_icon      = $is_custom_icon ? $selected_meta_icon : $default_icon;

					// Render the property meta with label, value, postfix (unit), icon, etc.
					rhea_render_single_property_meta_v2(
						$selected_meta_label,
						$meta_value,
						$postfix,
						$meta_key,
						$meta_icon,
						$is_custom_icon
					);
				}

				// Additional fields property meta.
				do_action( 'inspiry_additional_property_meta_fields', $property_id, array( 'custom_callback' => 'rhea_render_single_property_meta_v2' ) );
				?>
            </div>
        </div>
		<?php
	}
}