<?php
/**
 * Single Property Schedule A Tour Widget V2.
 *
 * @since 2.4.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Schedule_Tour_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	private static $property_id;

	public function __construct( $data = [], $args = null ) {
		parent::__construct( $data, $args );

		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}

		wp_register_script(
			'rh-single-property-schedule-tour-v2',
			RHEA_PLUGIN_URL . 'elementor/js/rh-single-property-schedule-tour-v2.js',
			[ 'elementor-frontend' ],
			RHEA_VERSION,
			true
		);

		wp_register_style(
			'jquery-ui-datepicker',
			'//code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css',
			array(),
			RHEA_VERSION,
            'all'
		);
	}

	public function get_name() {
		return 'rh-single-property-schedule-tour-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Schedule A Tour V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-calendar rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	public function get_script_depends() {
		return [ 'jquery-ui-datepicker', 'rh-single-property-schedule-tour-v2' ];
	}

	public function get_style_depends() {
		return [ 'jquery-ui-datepicker' ];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'sat_card_section',
			[
				'label' => esc_html__( 'Basic Settings', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'section_title',
			[
				'label'   => esc_html__( 'Section Title', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Schedule A Tour', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sat_labels',
			[
				'label'     => esc_html__( 'Labels', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'sta_date_label',
			[
				'label'   => esc_html__( 'Select Date Placeholder', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Select Date', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_in_person_label',
			[
				'label'   => esc_html__( 'In Person Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'In Person', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_video_chat_label',
			[
				'label'   => esc_html__( 'Video Chat Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Video Chat', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_name_label',
			[
				'label'   => esc_html__( 'Name Placeholder', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Your Name', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_phone_label',
			[
				'label'   => esc_html__( 'Phone Placeholder', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Your Phone', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_email_label',
			[
				'label'   => esc_html__( 'Email Placeholder', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Your Email', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_message_label',
			[
				'label'   => esc_html__( 'Message Placeholder', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Message', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->add_control(
			'sta_schedule_label',
			[
				'label'   => esc_html__( 'Submit Button Text', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Schedule', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->end_controls_section();

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'sat_container_styles',
			[
				'label' => esc_html__( 'Container', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'sat_container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_container_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-inner' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_container_items_row_gap',
			[
				'label'     => esc_html__( 'Items Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-inner' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'sat-container-bg',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-inner' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'sat_container_box_shadow',
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2-inner',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'sat_contact_form_styles',
			[
				'label' => esc_html__( 'Contact Form', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'sat_tour_type_section_heading',
			[
				'label'     => esc_html__( 'Tour Type Buttons', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'sat_tour_type_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_tour_type_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_tour_type_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type-inner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'sat_tour_type_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type input:checked+label' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_tour_type_border_color',
			[
				'label'     => esc_html__( 'Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type-inner' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_tour_type_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_tour_type_color_current',
			[
				'label'     => esc_html__( 'Current Button Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type input:checked+label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_tour_type_text_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2-tour-type label',
			]
		);

		$this->add_control(
			'sat_contact_form_fields_section_heading',
			[
				'label'     => esc_html__( 'Fields', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_fields_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form p,
					 {{WRAPPER}} .rh_inspiry_gdpr' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_fields_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form-field,
					 {{WRAPPER}} .rh-single-property-schedule-tour-v2 .bootstrap-select.rh-ultra-select-dropdown>.dropdown-toggle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_fields_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form-field, 
					 {{WRAPPER}} .rh-single-property-schedule-tour-v2 .bootstrap-select.rh-ultra-select-dropdown>.dropdown-toggle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_fields_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form-field,
					 {{WRAPPER}} .rh-inspiry-gdpr input:checked + label:before, 
					 {{WRAPPER}} .rh_inspiry_gdpr input:checked + label:before,
					 {{WRAPPER}} .comment-form-cookies-consent input:checked + label:before,
					 {{WRAPPER}} .rh-single-property-schedule-tour-v2 .bootstrap-select.rh-ultra-select-dropdown>.dropdown-toggle' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_fields_border_color',
			[
				'label'     => esc_html__( 'Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form-field,
					 {{WRAPPER}} .rh-single-property-schedule-tour-v2 .bootstrap-select.rh-ultra-select-dropdown>.dropdown-toggle,
					 {{WRAPPER}} .rh-single-property-schedule-tour-v2 .rh_inspiry_gdpr label:before' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_fields_text_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2-form-field, {{WRAPPER}} .bootstrap-select .dropdown-toggle .filter-option-inner-inner'
			]
		);

		$this->add_control(
			'sat_contact_form_fields_text_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form-field,
					 {{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown > .dropdown-toggle, 
					 {{WRAPPER}} .rh-inspiry-gdpr input:checked + label:before, 
					 {{WRAPPER}} .rh_inspiry_gdpr input:checked + label:before' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_fields_placeholder_color',
			[
				'label'     => esc_html__( 'Placeholder Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form ::-webkit-input-placeholder' => 'color: {{VALUE}}',
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form :-ms-input-placeholder'      => 'color: {{VALUE}}',
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-form ::placeholder'               => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'chevron_margin',
			[
				'label'     => esc_html__( 'Chevron Icon Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rhea-dropdown-style-chevron .bootstrap-select.inspiry_select_picker_trigger>.dropdown-toggle .caret:before' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'chevron_font_size',
			[
				'label'     => esc_html__( 'Chevron Icon Font Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-dropdown-style-chevron .bootstrap-select.inspiry_select_picker_trigger>.dropdown-toggle .caret:before' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_gdpr_title_section_heading',
			[
				'label'     => esc_html__( 'GDPR', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_gdpr_title_margin_bottom',
			[
				'label'     => esc_html__( 'Title Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .gdpr-checkbox-label' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_gdpr_title_typography',
				'label'    => esc_html__( 'Title Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2 .gdpr-checkbox-label',
			]
		);

		$this->add_control(
			'sat_contact_form_gdpr_title_color',
			[
				'label'     => esc_html__( 'Title Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .gdpr-checkbox-label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_gdpr_checkbox_margin',
			[
				'label'     => esc_html__( 'Checkbox Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh_inspiry_gdpr label:before' => 'Margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_gdpr_description_typography',
				'label'    => esc_html__( 'Description Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2 .rh-gdpr-text-inner-wrapper',
			]
		);

		$this->add_control(
			'sat_contact_form_gdpr_description_color',
			[
				'label'     => esc_html__( 'Description Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .rh-gdpr-text-inner-wrapper' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_submit_button_section_heading',
			[
				'label'     => esc_html__( 'Submit Button', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_submit_button_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_submit_button_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_submit_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button',
			]
		);

		$this->add_control(
			'sat_contact_form_submit_button_text_color',
			[
				'label'     => esc_html__( 'Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_submit_button_text_color_hover',
			[
				'label'     => esc_html__( 'Text Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_submit_button_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_submit_button_bg_hover_color',
			[
				'label'     => esc_html__( 'Background Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2 .submit-button:hover' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_error_color',
			[
				'label'     => esc_html__( 'Error Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .error-container label' => 'color: {{VALUE}}',
				],
				'separator' => 'before'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'calendar_colors_section',
			[
				'label' => esc_html__( 'Datepicker Calendar', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'datepicker_calendar_background_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div' => 'background-color: {{VALUE}}',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_heading-background_color',
			[
				'label'     => esc_html__( 'Heading Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-widget-header.ui-datepicker-header' => 'background-color: {{VALUE}} !important',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_heading_text_color',
			[
				'label'     => esc_html__( 'Heading color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-title' => 'color: {{VALUE}}',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_disabled_background_color',
			[
				'label'     => esc_html__( 'Disabled Dates Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar td.ui-state-disabled span' => 'background: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_disabled_border_color',
			[
				'label'     => esc_html__( 'Disabled Dates Border', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar td.ui-state-disabled span' => 'border-color: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_disabled_text_color',
			[
				'label'     => esc_html__( 'Disabled Dates Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar td.ui-state-disabled span' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_normal_background_color',
			[
				'label'     => esc_html__( 'Normal Dates Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div > table.ui-datepicker-calendar tbody tr td a' => 'background: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_normal_border_color',
			[
				'label'     => esc_html__( 'Normal Dates Border', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div > table.ui-datepicker-calendar tbody tr td a' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_normal_text_color',
			[
				'label'     => esc_html__( 'Normal Dates Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div > table.ui-datepicker-calendar tbody tr td a' => 'color: {{VALUE}}',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_normal_hover_background_color',
			[
				'label'     => esc_html__( 'Normal Dates Hover Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div > table.ui-datepicker-calendar tbody tr td a:hover' => 'background: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_normal_hover_border_color',
			[
				'label'     => esc_html__( 'Normal Dates Hover Border', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div > table.ui-datepicker-calendar tbody tr td a:hover' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_normal_hover_text_color',
			[
				'label'     => esc_html__( 'Normal Dates Hover Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div > table.ui-datepicker-calendar tbody tr td a:hover' => 'color: {{VALUE}}',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_today_background_color',
			[
				'label'     => esc_html__( 'Today Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar tbody tr td a.ui-state-highlight' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_today_border_color',
			[
				'label'     => esc_html__( 'Today Border', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar tbody tr td a.ui-state-highlight' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_today_text_color',
			[
				'label'     => esc_html__( 'Today Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar tbody tr td a.ui-state-highlight' => 'color: {{VALUE}}',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_active_background_color',
			[
				'label'     => esc_html__( 'Active Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar tbody tr td a.ui-state-active' => 'background-color: {{VALUE}};',
				]
			]
		);
		$this->add_control(
			'datepicker_calendar_active_border_color',
			[
				'label'     => esc_html__( 'Active Border', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar tbody tr td a.ui-state-active' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'datepicker_calendar_active_text_color',
			[
				'label'     => esc_html__( 'Active Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.rh-single-property-schedule-tour-v2-date-datepicker#ui-datepicker-div .ui-datepicker-calendar tbody tr td a.ui-state-active' => 'color: {{VALUE}}',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'sat_contact_form_time_dropdown',
			[
				'label' => esc_html__( 'Time Dropdown', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_select_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '.bootstrap-select .dropdown-toggle .filter-option-inner-inner',
			]
		);

		$this->add_control(
			'sat_contact_form_dropdown_color',
			[
				'label'     => esc_html__( 'Options Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_dropdown_active_color',
			[
				'label'     => esc_html__( 'Active Option Text', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li.selected .text'           => 'color: {{VALUE}}',
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li.selected span.check-mark' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_dropdown_hover_color',
			[
				'label'     => esc_html__( 'Options Hover Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select.rh-ultra-select-dropdown .dropdown-menu li:hover' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_dropdown_hover_text_color',
			[
				'label'     => esc_html__( 'Options Hover Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bootstrap-select .dropdown-menu li:hover a, 
					 {{WRAPPER}} .bootstrap-select .dropdown-menu li.selected:hover .text,
					 {{WRAPPER}} .bootstrap-select .dropdown-menu li.selected:hover span.check-mark' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_options_typography',
				'label'    => esc_html__( 'Options Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .bootstrap-select .dropdown-menu li a span.text',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'sat_contact_form_thumbnail',
			[
				'label' => esc_html__( 'Thumbnail and Content', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'show_thumbnail',
			[
				'label'        => esc_html__( 'Show Thumbnail', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_thumbnail_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-thumbnail img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'sat_contact_form_description_color',
			[
				'label'     => esc_html__( 'Description', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-description' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sat_contact_form_description_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-schedule-tour-v2-description',
			]
		);

		$this->add_responsive_control(
			'sat_contact_form_thumbnail_padding_top',
			[
				'label'     => esc_html__( 'Description Top Spacing', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-schedule-tour-v2-description' => 'padding-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {

		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() ) {
			return;
		}

		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();

		// getting schedule times for property
		$tour_times = get_post_meta( self::$property_id, 'ere_property_schedule_time_slots', true );
		if ( empty( $tour_times ) ) {
			// getting global schedule tour times in case the property specific are not set
			$tour_times = get_option( 'realhomes_schedule_time_slots' );
		}

		$tour_times_array     = explode( ',', $tour_times );
		$date_placeholder     = $settings['sta_date_label'];
		$in_person_label      = $settings['sta_in_person_label'];
		$video_chat_label     = $settings['sta_video_chat_label'];
		$name_placeholder     = $settings['sta_name_label'];
		$phone_placeholder    = $settings['sta_phone_label'];
		$email_placeholder    = $settings['sta_email_label'];
		$message_placeholder  = $settings['sta_message_label'];
		$schedule_button_text = $settings['sta_schedule_label'];
		$schedule_description = get_post_meta( self::$property_id, 'ere_property_schedule_description', true );
		$user_name            = $user_email = $user_phone = '';

		if ( empty( $schedule_description ) ) {
			$schedule_description = get_option( 'realhomes_schedule_side_description' );
		}

		if ( is_user_logged_in() ) {
			$current_user = wp_get_current_user();
			$user_name    = ! empty( $current_user->display_name ) ? $current_user->display_name : $current_user->user_login;
			$user_email   = $current_user->user_email;
			$user_phone   = get_user_meta( $current_user->ID, 'mobile_number', true );
		}
		?>
        <div id="rh-single-property-schedule-tour-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-schedule-tour-v2">
			<?php
			if ( ! empty( $settings['section_title'] ) ) {
				?>
                <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
				<?php
			}
			?>
            <div class="rh-single-property-schedule-tour-v2-inner">
                <div class="rh-single-property-schedule-tour-v2-col rh-single-property-schedule-tour-v2-col-left">
                    <form id="rh-single-property-schedule-tour-v2-form" class="rh-single-property-schedule-tour-v2-form" method="post" action="<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>">
                        <div class="rh-single-property-schedule-tour-v2-tour-type">
                            <div class="rh-single-property-schedule-tour-v2-tour-type-inner">
                                <div>
                                    <input type="radio" id="sat-in-person" name="sat-tour-type" value="<?php echo esc_attr( $in_person_label ); ?>" checked>
                                    <label for="sat-in-person"><?php echo esc_attr( $in_person_label ); ?></label>
                                </div>
                                <div>
                                    <input type="radio" id="sat-video-chat" name="sat-tour-type" value="<?php echo esc_attr( $video_chat_label ); ?>">
                                    <label for="sat-video-chat"><?php echo esc_attr( $video_chat_label ); ?></label>
                                </div>
                            </div>
                        </div>
                        <div class="rh-single-property-schedule-tour-v2-col-wrapper rhea-dropdown-style-chevron">
                            <p>
                                <input type="text" name="sat-date" id="schedule-date-<?php echo esc_attr( $widget_id ); ?>" class="required rh-single-property-schedule-tour-v2-form-field" placeholder="<?php echo esc_attr( $date_placeholder ); ?>" title="<?php esc_attr_e( '* Select a suitable date', RHEA_TEXT_DOMAIN ); ?>" autocomplete="off" required>
                            </p>
							<?php
							if ( is_array( $tour_times_array ) && ! empty( $tour_times_array ) ) {
								?>
                                <p>
                                    <select name="sat-time" class="sat_times rh-ultra-select-dropdown inspiry_select_picker_trigger show-tick">
										<?php
										foreach ( $tour_times_array as $tour_time ) {
											echo '<option value="' . esc_attr( $tour_time ) . '">' . esc_html( $tour_time ) . '</option>' . PHP_EOL;
										}
										?>
                                    </select>
                                </p>
								<?php
							}
							?>
                        </div>
                        <div class="rh-single-property-schedule-tour-v2-col-wrapper">
                            <p>
                                <input id="sat-user-name" type="text" name="sat-user-name" class="required rh-single-property-schedule-tour-v2-form-field" placeholder="<?php echo esc_attr( $name_placeholder ); ?>" value="<?php echo esc_attr( $user_name ); ?>" title="<?php esc_attr_e( '* Provide your name', RHEA_TEXT_DOMAIN ); ?>" required>
                            </p>
                            <p>
                                <input id="sat-user-email" type="text" name="sat-user-email" class="required rh-single-property-schedule-tour-v2-form-field" placeholder="<?php echo esc_attr( $email_placeholder ); ?>" value="<?php echo esc_attr( $user_email ); ?>" title="<?php esc_attr_e( '* Provide your email ID', RHEA_TEXT_DOMAIN ); ?>" required>
                            </p>
                        </div>
                        <p>
                            <input id="sat-user-phone" type="text" name="sat-user-phone" class="rh-single-property-schedule-tour-v2-form-field" placeholder="<?php echo esc_attr( $phone_placeholder ); ?>" value="<?php echo esc_attr( $user_phone ); ?>">
                        </p>
                        <p>
                            <textarea id="sat-user-message" class="rh-single-property-schedule-tour-v2-form-field" cols="40" rows="6" name="sat-user-message" placeholder="<?php echo esc_attr( $message_placeholder ); ?>"></textarea>
                        </p>
						<?php
						if ( function_exists( 'ere_gdpr_agreement' ) ) {
							ere_gdpr_agreement( array(
								'id'              => 'rh_inspiry_gdpr',
								'container_class' => 'rh_inspiry_gdpr',
								'title_class'     => 'gdpr-checkbox-label'
							) );
						}

						if ( class_exists( 'Easy_Real_Estate' ) ) {
							/* Display reCAPTCHA if enabled and configured from customizer settings */
							if ( ere_is_reCAPTCHA_configured() ) {
								$recaptcha_type = get_option( 'inspiry_reCAPTCHA_type', 'v2' );
								?>
                                <div class="rh_contact__input rh_contact__input_recaptcha inspiry-recaptcha-wrapper clearfix g-recaptcha-type-<?php echo esc_attr( $recaptcha_type ); ?>">
                                    <div class="inspiry-google-recaptcha"></div>
                                </div>
								<?php
							}
						}
						?>
                        <input type="hidden" name="action" value="schedule_a_tour" />
                        <input type="hidden" name="property-id" value="<?php echo esc_attr( self::$property_id ); ?>" />
                        <input type="hidden" name="sat-nonce" value="<?php echo esc_attr( wp_create_nonce( 'schedule_a_tour_nonce' ) ); ?>" />
                        <button type="submit" name="submit" class="submit-button">
                            <span class="btn-text"><?php echo esc_attr( $schedule_button_text ); ?></span>
                            <span id="ajax-loader-<?php echo esc_attr( $widget_id ); ?>" class="ajax-loader"><?php inspiry_safe_include_svg( '/icons/ball-triangle.svg' ); ?></span>
                        </button>
                        <div class="error-container"></div>
                        <div class="message-container"></div>
                    </form>
                </div>
				<?php
				if ( ! empty( $settings['show_thumbnail'] ) ) {
					?>
                    <div class="rh-single-property-schedule-tour-v2-col rh-single-property-schedule-tour-v2-col-right">
                        <div class="rh-single-property-schedule-tour-v2-thumbnail">
							<?php
							if ( has_post_thumbnail( self::$property_id ) ) {
								echo get_the_post_thumbnail( self::$property_id, 'property-detail-video-image' );
							}
							?>
                        </div>
                        <div class="rh-single-property-schedule-tour-v2-description">
							<?php echo wp_kses_post( $schedule_description ); ?>
                        </div>
                    </div>
					<?php
				}
				?>
            </div>
        </div>
		<?php
		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			?>
            <script type="application/javascript">
                rheaSelectPicker( '.inspiry_select_picker_trigger' );
            </script>
			<?php
		}
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}
}